#!/usr/bin/env python

# Capstone Python bindings, by Peace-Maker <peacemakerctf@gmail.com>

from __future__ import print_function
from capstone import *
from capstone.sh import *
from xprint import to_x, to_hex

SH4A_CODE = b"\x0c\x31\x10\x20\x22\x21\x36\x64\x46\x25\x12\x12\x1c\x02\x08\xc1\x05\xc7\x0c\x71\x1f\x02\x22\xcf\x06\x89\x23\x00\x2b\x41\x0b\x00\x0e\x40\x32\x00\x0a\xf1\x09\x00"
SH2A_CODE = b"\x32\x11\x92\x00\x32\x49\x31\x00"
all_tests = (
        (CS_ARCH_SH, CS_MODE_SH4A | CS_MODE_SHFPU, SH4A_CODE, "SH_SH4A"),
        (CS_ARCH_SH, CS_MODE_SH2A | CS_MODE_SHFPU | CS_MODE_BIG_ENDIAN, SH2A_CODE, "SH_SH2A"),
)


reg_address_msg = [
	"Register indirect",
	"Register indirect with predecrement",
	"Register indirect with postincrement",
]

def print_read_write_regs(insn):
    if len(insn.regs_read) > 0:
        print("\tRegisters read: %s" % " ".join(insn.reg_name(m) for m in insn.regs_read))

    if len(insn.regs_write) > 0:
        print("\tRegisters modified: %s" % " ".join(insn.reg_name(m) for m in insn.regs_write))

def print_insn_detail(insn):
    # print address, mnemonic and operands
    print("0x%x:\t%s\t%s" % (insn.address, insn.mnemonic, insn.op_str))

    # "data" instruction generated by SKIPDATA option has no detail
    if insn.id == 0:
        return

    if len(insn.operands) > 0:
        print("\top_count: %u" % len(insn.operands))
        c = 0
        for i in insn.operands:
            if i.type == SH_OP_REG:
                print("\t\toperands[%u].type: REG = %s" % (c, insn.reg_name(i.reg)))
            elif i.type == SH_OP_IMM:
                print("\t\toperands[%u].type: IMM = 0x%s" % (c, to_x(i.imm)))
            elif i.type == SH_OP_MEM:
                print("\t\toperands[%u].type: MEM " % c)
                if i.mem.address in [SH_OP_MEM_REG_IND, SH_OP_MEM_REG_POST, SH_OP_MEM_REG_PRE]:
                    print("%s REG %s" % (reg_address_msg[i.mem.address - SH_OP_MEM_REG_IND], insn.reg_name(i.mem.reg)))
                elif i.mem.address == SH_OP_MEM_REG_DISP:
                    print("Register indirect with displacement REG %s, DISP %d" % (insn.reg_name(i.mem.reg), i.mem.disp))
                elif i.mem.address == SH_OP_MEM_REG_R0:
                    print("R0 indexed")
                elif i.mem.address == SH_OP_MEM_GBR_DISP:
                    print("GBR base with displacement DISP %d" % i.mem.disp)
                elif i.mem.address == SH_OP_MEM_GBR_R0:
                    print("GBR base with R0 indexed")
                elif i.mem.address == SH_OP_MEM_PCR:
                    print("PC relative Address=0x%08x" % i.mem.disp)
                elif i.mem.address == SH_OP_MEM_TBR_DISP:
                    print("TBR base with displacement DISP %d", i.mem.disp)
                else:
                    print("Unknown addressing mode %x" % i.mem.address)
            
            if i.sh_size != 0:
                print("\t\t\tsh_size: %u" % i.sh_size)
            c += 1

    print_read_write_regs(insn)

    if len(insn.groups) > 0:
        print('\tgroups: ' + ' '.join(map(lambda g: insn.group_name(g), insn.groups)))


# ## Test class Cs
def test_class():

    for (arch, mode, code, comment) in all_tests:
        print("*" * 16)
        print("Platform: %s" %comment)
        print("Code: %s" % to_hex(code))
        print("Disasm:")

        try:
            md = Cs(arch, mode)
            md.detail = True
            for insn in md.disasm(code, 0x80000000):
                print_insn_detail(insn)
                print()
            print("0x%x:" % (insn.address + insn.size))
        except CsError as e:
            print("ERROR: %s" %e)


if __name__ == '__main__':
    test_class()
